#!/usr/bin/env python3
# This file is part of granularhealing.

# granularhealing is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# granularhealing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with granularhealing. If not, see <https://www.gnu.org/licenses/>.
"""
Functions for calculating stress angles and reactivation angles from
frictional data.

Uses uncertainties for error propagation.

"""

import numpy as np
from uncertainties import unumpy as unp


def fric_ang(mu):
    """Gets friction angle phi from friction mu"""
    return rad2deg(unp.tan(mu))


def theta_ang(phi):
    """Gets angle theta from friction angle phi"""
    return 45 + 0.5 * phi


def fault_ang(phi):
    """Gets angle alpha to sigma1"""
    return 45 - 0.5 * phi


def active_fault_angle(mu):
    """
    Angle forming in extension according to Coulomb analysis after Verruijt
    2018
    """
    return rad2deg(0.25 * np.pi - 0.5 * unp.tan(mu))


def passive_fault_angle(mu):
    """
    Angle forming in compression according to Coulomb analysis after Verruijt
    2018
    """
    return rad2deg(0.25 * np.pi + 0.5 * unp.tan(mu))


def weight_active(rho, h, phi):
    """
    Weight of wedge in active case after Verruijt 2018
    """
    return 0.5 * rho * 9.81 * h**2 * unp.tan(unp.radians(phi))


def react_ratio(delta, mu, lenratio):
    """
    Calculates the ratio between tectonic and gravitational stress for
    reactivation of faults. After: Mulugeta and Sokoutis, 2003

    Parameters:
        - delta: angle of fault to be reactivated
        - mu: friction on fault
        - lenratio: ratio of length to height (L/z)
    """
    delta = deg2rad(delta)
    a = mu * lenratio
    b = ((1 + mu * (1 / cot(delta))) / (1 - mu * unp.tan(delta))) - 1
    return a + 0.5 * b


def opt_angle(mu):
    """Optimal angle for fault reactivation"""
    return rad2deg(0.5 * unp.arctan(1 / mu))


def lock_angle(mu):
    """Angle of full locking"""
    return 2 * opt_angle(mu)


def react_ratio_normal(delta, mu):
    """
    Reactivation of a normal fault according to Collettini et al. 2001
    """
    delta = unp.radians(delta)
    a = mu * (unp.tan(delta) + cot(delta))
    b = 1 + mu * cot(delta)
    return a / b


def react_ratio_reverse(delta, mu):
    """
    Reactivation of a reverse fault according to Collettini et al. 2001
    """
    delta = unp.radians(delta)
    a = mu * (unp.tan(delta) + cot(delta))
    b = 1 - mu * unp.tan(delta)
    return a / b


def active_pressure_coulomb(h, mu, theta, coh, rho):
    """
    Pressure on a side wall during extension after Coulomb according to
    Verruijt 2018
    """
    phi = unp.arctan(mu)
    theta = deg2rad(theta)
    gamma = rho * 9.81
    a = 0.5 * gamma * h**2
    b = a * unp.sin(phi) + coh * h * unp.cos(phi)
    c = unp.cos(theta) * unp.sin(theta + phi)
    q = a - b / c
    return q


def passive_pressure_coulomb(h, mu, theta, coh, rho):
    """
    Pressure on a side wall during compression after Coulomb according to
    Verruijt 2018
    """
    phi = unp.arctan(mu)
    theta = deg2rad(theta)
    gamma = rho * 9.81
    a = 0.5 * gamma * h**2
    b = a * unp.sin(phi) + coh * h * unp.cos(phi)
    c = unp.cos(theta) * unp.sin(theta - phi)
    q = a + b / c

    return q


def active_coeff_rankine(mu):
    """
    Active earth pressure coefficient
    """
    phi = unp.arctan(mu)
    ka = (1 - unp.sin(phi)) / (1 + unp.sin(phi))
    return ka


def passive_coeff_rankine(mu):
    """
    Passive earth pressure coefficient
    """
    phi = unp.arctan(mu)
    kp = (1 + unp.sin(phi)) / (1 - unp.sin(phi))
    return kp


def force_triangle(h, mu, theta, rho):
    """
    Simple Triangular Forces without cohesion
    """
    phi = unp.arctan(mu)
    theta = deg2rad(theta)
    gamma = rho * 9.81
    e = 0.5 * ((gamma * h**2) / unp.tan(theta)) * unp.tan(theta + phi)

    return e


def force_test(h, mu, theta_in, coh, rho, side=0.1):
    """Force a triangle uphill"""
    g = 9.81
    theta = deg2rad(theta_in)
    theta2 = deg2rad(90 - theta_in)
    area = 0.5 * h * (h / unp.cos(theta)) * unp.sin(theta)
    # print(theta_in, "A=", area)
    volume = area * side
    mass = volume * rho
    Fg = mass * g
    Fgn = Fg * unp.cos(theta2)
    # Fgh = Fg * unp.sin(theta)
    T = mu * Fgn + coh
    return T


def cot(a):
    """
    Returns the Cotangens (1/tan)
    """
    return 1 / unp.tan(a)


def rad2deg(a):
    return a * (180 / np.pi)


def deg2rad(a):
    return (a / 180) * np.pi


def force_mulugeta(mu, angle_in, h, rho):
    angle = deg2rad(angle_in)
    a = (rho * 9.81 * h**2 * cot(angle)) / 2
    b = (mu + unp.tan(angle)) / (1 - mu * unp.tan(angle))
    result = a * b
    return result


def ratio_mulugeta(mu, angle_in, lenratio):
    angle = deg2rad(angle_in)
    a = mu * lenratio
    b = 1 + mu * cot(angle)
    c = 1 - mu * unp.tan(angle)
    result = a + 0.5 * (b / c - 1)
    return result
