#!/usr/bin/env python3
# This file is part of granularhealing.

# granularhealing is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# granularhealing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with granularhealing. If not, see <https://www.gnu.org/licenses/>.

""" Functions for handling granularhealing configs """

import configparser


def get_config(cfg_file_path: str) -> configparser.ConfigParser:
    """
    Reads a configuration file from `file_path` or creates a new default config
    if none was found.
    """
    cfg = configparser.ConfigParser(create_default())
    cfg_from_file = configparser.ConfigParser()
    cfg.read_dict(create_default())
    try:
        cfg_from_file.read(cfg_file_path)
        cfg.update(cfg_from_file)
    finally:
        save_config(cfg, cfg_file_path)

    return cfg


def create_default_config() -> configparser.ConfigParser:
    """
    Creates default config file for processing the granular healing data.
    """
    cfg_dict = create_default()

    cfg = configparser.ConfigParser()
    cfg.read_dict(cfg_dict)

    return cfg


def create_default() -> dict:
    """Default config as dictionary"""
    main_opts = {
        # MAIN OPTIONS
        # Overwrites result files
        "overwrite": True,
    }

    pick_opts = {
        # PICKING
        # Velocity threshold
        "vlim": 0.01,
        # Timeframe to pick value (seconds)
        "post_region": 1.0,
        # Shift of pick region start (seconds)
        "post_shift": 0.0,
        # Time frame for calculating steady state friction (seconds)
        # if 0 then all values up to next hold or end of file are used.
        "dyn_region": 0.0,
        # minimum hold time (seconds)
        "min_thold": 5.0,
        # Time region for area before reloading (seconds)
        "pre_loading": 1.0,
        # Time region for area just after stopping (seconds)
        "post_stop": 2.0,
    }

    cfg_dict = {"main": main_opts, "pick": pick_opts}

    return cfg_dict


def save_config(cfg: configparser.ConfigParser, cfg_file_path: str):
    """Tries to save the config to the given file path."""
    try:
        with open(cfg_file_path, "w") as cfg_file:
            cfg.write(cfg_file)
    except OSError as error:
        raise error
