#!/usr/bin/env python3
# This file is part of granularhealing.

# granularhealing is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# granularhealing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with granularhealing. If not, see <https://www.gnu.org/licenses/>.
""" Group associations and other info for each dataset """
import numpy as np


def get_group_dict() -> dict:
    """Contains a group association for all datasets"""

    mats = [
        "quartz sand",
        "corundum sand",
        "feldspar sand",
        "garnet sand",
        "glass beads",
        "zircon sand",
        "foam glass",
    ]

    mat_group = {
        "02_BernNeusand_1kPa": 0,
        "03_UtrechtSand2018_1kPa": 0,
        "04_BernNeusand_20kPa": 0,
        "05_RennesSandNE342021_1kPa": 0,
        "06_ParmaSand_1kPa": 0,
        "07_KyotoSand_1kPa": 0,
        "08_BernSand2008_1kPa": 0,
        "09_GFZKorundNKF120_1kPa": 1,
        "10_UppsalaSand_1kPa": 0,
        "11_CergyPontoiseSand_1kPa": 0,
        "12_KorundsandBern2019_1kPa": 1,
        "13_FeldspatsandUtrecht_1kPa": 2,
        "14_QuartzsandRHULondon_1kPa": 0,
        "15b_G12GFZsmaller400_1kPa": 0,
        "15_G12GFZsmaller400_1kPa": 0,
        "16_G23GFZsmaller630_1kPa": 0,
        "17_QuartzsandPragST55_1kPa": 0,
        "18_GarnetSandGFZ2008_1kPa": 3,
        "19_Glasperlen70-110GFZ_1kPa": 4,
        "20_Glasperlen300-400_1kPa": 4,
        "21_Glasperlen40-70GFZ_1kPa": 4,
        "22_Glasperlen100-200GFZ_1kPa": 4,
        "23_ZirkonsandGFZ2009_1kPa": 5,
        "24_IFPKorundWeissBern2008_1kPa": 1,
        "25_GlasbeadsPrag0-50poured_1kPa": 4,
        "26_GlasbeadsPrag100-200_1kPa": 4,
        "28_SandBernCarloAG100-300_1kPa": 0,
        "29_BernNeusand2015_5kPa": 0,
        "30_BernNeusand2015_10kPa": 0,
        "31_BernNeusand2015_15kPa": 0,
        "35_FS900SBernFspSand_1kPa": 2,
        "36_LilleQtzSandG39_1kPa": 0,
        "37_LilleQtzSandNE34_1kPa": 0,
        "38_LilleGBCVP70-110_1kPa": 4,
        "39_LilleGBEyraud70-110_1kPa": 4,
        "40_PragueQtzSand_1kPa": 0,
        "41_WroclawQtzSand_1kPa": 0,
        "42_TaiwanQtzSand_1kPa": 0,
        "43_OuroPretoQtzSand_1kPa": 0,
        "45_BernKorundSand_1kPa": 1,
        "46_FreiburgQtzSand_1kPa": 0,
        "47b_GFZFoamglass250-500_1kPa": 6,
        "47_GFZFoamglass250-500_1kPa": 6,
        "48_BernNeusand_1kPa": 0,
        "49_GFZFoamglass100-300_1kPa": 6,
        "50_BernNeusand_15kPa": 0,
        "071-01_Korund_B_MK": 1,
        "101-01_Granatsand_GFZ": 3,
        "102-01_PB_Bernsand": 0,
        "103-01_PB_Lille1": 0,
        "105-01_PB_Taiwan1": 0,
        "108-01_PB_Cergy_Pont": 0,
        "109-01_PB_Royal_S": 0,
        "110-01_PB_Uppsala": 0,
        "111-01_PB_Kyoto_S": 0,
        "112-01_PB_Ouro_Preto": 0,
        "113-01_PB_Parma_S": 0,
        "114-01_PB_Bernsand2": 0,
        "115-01_PB_Bernsand3": 0,
        "129-01_PB_GFZ_Sand": 0,
        "137-01_PB_IFP_C_wi": 1,
        "140-01_PB_IFP_C_br": 1,
        "150-01 Bernsand": 0,
        "153-01_PB_Pragsand": 0,
        "155-01_PB_Wroclaws": 0,
        "159-01_PB_Zirkon": 5,
        "163-01_PB_GfzGran": 3,
        "165-01_PB_Glb_03-04": 4,
        "167-01_PB_Glb_01-02": 4,
        "169-01_PB_Glb_01-04": 4,
        "176-01_G23(2009)": 0,
        "177-01_G23(2009)": 0,
        "178-01_G23(2009)": 0,
        "185-01 G23 2005": 0,
        "186-01-G23(2005)": 0,
        "187-01 G23(2005)": 0,
        "190-01 Glassbeads 200-300": 4,
        "191-01 Glassbeads 200-300": 4,
        "192-01 Glassbeads 200-300": 4,
        "193-01 Glassbeads 40-70": 4,
        "194-01 Glassbeads 40-70": 4,
        "195-01 Glassbeads 40-70": 4,
        "196-01 Glassbeads 100-200": 4,
        "197-01 Glassbeads 70-110": 4,
        "198-01 Glassbeads 100-200": 4,
        "222-01_G23Sand2010": 0,
        "223-01_G23Sand2009": 0,
        "224-01_Glassbeads300-400": 4,
        "226-01_GB100-200": 4,
        "227-01_GB100-200": 4,
        "228-01_GB300-400": 4,
        "230-01_GB300-400": 4,
        "231-01_GB70-110": 4,
        "232-01_GB70-110": 4,
        "233-01_GB40-70": 4,
        "242-01 GB 40-70": 4,
        "243-01 GB 200-300": 4,
        "244-01 GB 200-300": 4,
        "251-01 GB 40-70": 4,
        "252-01 GB 70-110": 4,
        "253-01 GB 100-200": 4,
        "254-01 GB 200-300": 4,
        "255-01 GB 300-400": 4,
        "270-01 Granatsand": 3,
        "281-01_VUBSand_SamPoppe": 0,
        "286-01_VUB-Sand_Sieved_SamPoppe": 0,
        "323-01_G23": 0,
        "352-01_Glasperlen_100-200": 4,
        "355-01_Glasperlen_40-70": 4,
        "356-01_Glasperlen_200-300": 4,
        "357_01_Glasperlen_300-400": 4,
        "362-01_Glasperlen_70-110": 4,
        "373-01_DB2017-X1_40-60_0.3-0.45mm": 0,
        "374-01_DB2017-X2_40-60_0.2-0.3mm": 0,
        "377-01 DB2017-L1_40-60_0.3-0.45mm": 0,
        "378-01_DB2017-L2_60-80_0.2-0.3mm": 0,
        "385-01_Zaragoza_Quartz_Sand": 0,
        "390-01_Feldspar_sand_Utrecht": 2,
        "391-01_Quartz_sand_Utrecht": 0,
        "392-01_Foamglas_0,25-0,5": 6,
        "393-01_Foamglas_0,1-0,3": 6,
        "399-01_G12": 0,
        "400-01_G23": 0,
        "419-01_Quartzsand_Zaragosa_L-7080S": 0,
        "424-01_G12": 0,
        "426-01_G23": 0,
        "431-01_Qtz-Sand Fort_Rennes": 0,
        "432-01_VUB_Sand_Belgium": 0,
        "435-01_Quartz_Sand_Bern": 0,
        "436-01_Korundsand_NFK120": 1,
        "437-01_Glassbeads_300_400": 4,
        "439-01_Glassbeads_200_300": 4,
        "440-01_Granatsand": 3,
        "441-01_Glassbeads_100_200": 4,
        "442-01_Glassbeads_40_70": 4,
        "443-03_Bernsand_30cm": 0,
        "444-03_Bernkorund_30cm": 1,
        "446-01_Glassbeads_70_110": 4,
        "452-01_Quartzsand_ST55_Prague": 0,
        "467-01_Glassbeads_200_300": 4,
        "468-01_Feldsparsand_Florence": 2,
        "491-01_Glassbeads_200_300_Prague": 4,
        "494-01_Quartzsand_Uni_Freiburg": 0,
    }

    annot = {
        "02_BernNeusand_1kPa": "Bern sand (2015)",
        "03_UtrechtSand2018_1kPa": "Utrecht Sand (stick slips)",
        "04_BernNeusand_20kPa": "Bern 20kPa (2015)",
        "05_RennesSandNE342021_1kPa": "Rennes, NE34",
        "06_ParmaSand_1kPa": "Parma",
        "07_KyotoSand_1kPa": "Kyoto",
        "08_BernSand2008_1kPa": "Bern sand (2008)",
        "09_GFZKorundNKF120_1kPa": "GFZ, NKF120",
        "10_UppsalaSand_1kPa": "Uppsala",
        "11_CergyPontoiseSand_1kPa": "Cergy Pontoise",
        "12_KorundsandBern2019_1kPa": "Bern, brown (2019)",
        "13_FeldspatsandUtrecht_1kPa": "Utrecht",
        "14_QuartzsandRHULondon_1kPa": "RHU London",
        "15b_G12GFZsmaller400_1kPa": "GFZ, GB (G12 <400µm)",
        "15_G12GFZsmaller400_1kPa": "GFZ, GB (G12 <400µm)",
        "16_G23GFZsmaller630_1kPa": "GFZ, GB (G23 <630µm)",
        "17_QuartzsandPragST55_1kPa": "Prag",
        "18_GarnetSandGFZ2008_1kPa": "GFZ Garnet",
        "19_Glasperlen70-110GFZ_1kPa": "GFZ, GB (70-110µm)",
        "20_Glasperlen300-400_1kPa": "GFZ, GB (300-400µm)",
        "21_Glasperlen40-70GFZ_1kPa": "GFZ, GB (40-70µm)",
        "22_Glasperlen100-200GFZ_1kPa": "GFZ, GB (100-200µm)",
        "23_ZirkonsandGFZ2009_1kPa": "GFZ Zircon",
        "24_IFPKorundWeissBern2008_1kPa": "Bern, white (2008)",
        "25_GlasbeadsPrag0-50poured_1kPa": "Prag, GB (0-50µm)",
        "26_GlasbeadsPrag100-200_1kPa": "Prag, GB (100-200µm)",
        "28_SandBernCarloAG100-300_1kPa": "Bern, CarloAG (100-300µm)",
        "29_BernNeusand2015_5kPa": "Bern, 5kPa (2015)",
        "30_BernNeusand2015_10kPa": "Bern, 10kPa (2015)",
        "31_BernNeusand2015_15kPa": "Bern, 15kPa (2015)",
        "35_FS900SBernFspSand_1kPa": "Bern, FS900S",
        "36_LilleQtzSandG39_1kPa": "Lille, GA39",
        "37_LilleQtzSandNE34_1kPa": "Lille, NE34",
        "38_LilleGBCVP70-110_1kPa": "Lille, CVP (70-110µm)",
        "39_LilleGBEyraud70-110_1kPa": "Lille, Eyraud (70-110µm)",
        "40_PragueQtzSand_1kPa": "Prag",
        "41_WroclawQtzSand_1kPa": "Wroclaw",
        "42_TaiwanQtzSand_1kPa": "Taiwan",
        "43_OuroPretoQtzSand_1kPa": "Ouro Preto",
        "45_BernKorundSand_1kPa": "Bern, brown (2008)",
        "46_FreiburgQtzSand_1kPa": "Freiburg",
        "47b_GFZFoamglass250-500_1kPa": "GFZ, GB (250-500µm)",
        "392-01_Foamglas_0,25-0,5": "GFZ, GB (250-500µm)",
        "393-01_Foamglas_0,1-0,3": "GFZ, GB (100-300µm)",
        "47_GFZFoamglass250-500_1kPa": "GFZ, GB (250-500µm)",
        "48_BernNeusand_1kPa": "Bern sand (2015)",
        "49_GFZFoamglass100-300_1kPa": "GFZ, GB (100-300µm)",
        "50_BernNeusand_15kPa": "Bern, 15 kPa (2015)",
        "071-01_Korund_B_MK": "Bern corundum brown",
        "101-01_Granatsand_GFZ": "GFZ garnet sand",
        "102-01_PB_Bernsand": "Bern sand",
        "103-01_PB_Lille1": "Lille sand",
        "105-01_PB_Taiwan1": "Taiwan sand",
        "108-01_PB_Cergy_Pont": "Cergy-Pontoise sand",
        "109-01_PB_Royal_S": "Royal-Holloway sand",
        "110-01_PB_Uppsala": "Uppsala sand",
        "111-01_PB_Kyoto_S": "Kyoto sand",
        "112-01_PB_Ouro_Preto": "Ouro-Preto sand",
        "113-01_PB_Parma_S": "Parma sand",
        "114-01_PB_Bernsand2": "Bern sand 2",
        "115-01_PB_Bernsand3": "Bern sand 3",
        "129-01_PB_GFZ_Sand": "GFZ sand",
        "137-01_PB_IFP_C_wi": "IFP corundum sand white",
        "140-01_PB_IFP_C_br": "IFP corundum sand brown",
        "150-01 Bernsand": "Bern sand",
        "153-01_PB_Pragsand": "Prague sand",
        "155-01_PB_Wroclaws": "Wroclaw sand",
        "159-01_PB_Zirkon": "GFZ zircon sand",
        "163-01_PB_GfzGran": "GFZ garnet sand",
        "165-01_PB_Glb_03-04": "GFZ, GB (300-400µm)",
        "167-01_PB_Glb_01-02": "GFZ, GB (100-200µm)",
        "169-01_PB_Glb_01-04": "GFZ, GB (100-400µm)",
        "176-01_G23(2009)": "GFZ G23 sand",
        "177-01_G23(2009)": "GFZ G23 sand",
        "178-01_G23(2009)": "GFZ G23 sand",
        "185-01 G23 2005": "GFZ G23 sand",
        "186-01-G23(2005)": "GFZ G23 sand",
        "187-01 G23(2005)": "GFZ G23 sand",
        "190-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "191-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "192-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "193-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "194-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "195-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "196-01 Glassbeads 100-200": "GFZ, GB (100-200µm)",
        "197-01 Glassbeads 70-110": "GFZ, GB (100-200µm)",
        "198-01 Glassbeads 100-200": "GFZ, GB (100-200µm)",
        "222-01_G23Sand2010": "GFZ G23 sand 2010",
        "223-01_G23Sand2009": "GFZ G23 sand 2009",
        "224-01_Glassbeads300-400": "GFZ, GB (300-400µm)",
        "226-01_GB100-200": "GFZ, GB (100-200µm)",
        "227-01_GB100-200": "GFZ, GB (100-200µm)",
        "228-01_GB300-400": "GFZ, GB (300-400µm)",
        "230-01_GB300-400": "GFZ, GB (300-400µm)",
        "231-01_GB70-110": "GFZ, GB (70-110µm)",
        "232-01_GB70-110": "GFZ, GB (70-110µm)",
        "233-01_GB40-70": "GFZ, GB (40-70µm)",
        "242-01 GB 40-70": "GFZ, GB (40-70µm)",
        "243-01 GB 200-300": "GFZ, GB (200-300µm)",
        "244-01 GB 200-300": "GFZ, GB (200-300µm)",
        "251-01 GB 40-70": "GFZ, GB (40-70µm)",
        "252-01 GB 70-110": "GFZ, GB (70-110µm)",
        "253-01 GB 100-200": "GFZ, GB (100-200µm)",
        "254-01 GB 200-300": "GFZ, GB (200-300µm)",
        "255-01 GB 300-400": "GFZ, GB (300-400µm)",
        "270-01 Granatsand": "GFZ garnet sand",
        "281-01_VUBSand_SamPoppe": "VUB sand",
        "286-01_VUB-Sand_Sieved_SamPoppe": "VUB sand",
        "323-01_G23": "GFZ G23 sand",
        "352-01_Glasperlen_100-200": "GFZ, GB (100-200µm)",
        "355-01_Glasperlen_40-70": "GFZ, GB (40-70µm)",
        "356-01_Glasperlen_200-300": "GFZ, GB (200-300µm)",
        "357_01_Glasperlen_300-400": "GFZ, GB (300-400µm)",
        "362-01_Glasperlen_70-110": "GFZ, GB (70-110µm)",
        "373-01_DB2017-X1_40-60_0.3-0.45mm": "Chengdu X1",
        "374-01_DB2017-X2_40-60_0.2-0.3mm": "Chengdu X2",
        "377-01 DB2017-L1_40-60_0.3-0.45mm": "Chengdu L1",
        "378-01_DB2017-L2_60-80_0.2-0.3mm": "Chengdu L2",
        "385-01_Zaragoza_Quartz_Sand": "Zaragoza sand",
        "390-01_Feldspar_sand_Utrecht": "Utrecht feldspar sand",
        "391-01_Quartz_sand_Utrecht": "Utrecht sand",
        "399-01_G12": "GFZ G12 sand",
        "400-01_G23": "GFZ G23 sand",
        "419-01_Quartzsand_Zaragosa_L-7080S": "Zaragoza sand 7080",
        "424-01_G12": "GFZ G12 sand",
        "426-01_G23": "GFZ G23 sand",
        "431-01_Qtz-Sand Fort_Rennes": "Rennes sand",
        "432-01_VUB_Sand_Belgium": "VUB sand",
        "435-01_Quartz_Sand_Bern": "Bern sand",
        "436-01_Korundsand_NFK120": "NFK120 corundum sand",
        "437-01_Glassbeads_300_400": "GFZ, GB (300-400µm)",
        "439-01_Glassbeads_200_300": "GFZ, GB (200-300µm)",
        "440-01_Granatsand": "GFZ garnet sand",
        "441-01_Glassbeads_100_200": "GFZ, GB (100-200µm)",
        "442-01_Glassbeads_40_70": "GFZ, GB (40-70µm)",
        "443-03_Bernsand_30cm": "Bern sand",
        "444-03_Bernkorund_30cm": "Bern corundum sand",
        "446-01_Glassbeads_70_110": "GFZ, GB (70-110µm)",
        "452-01_Quartzsand_ST55_Prague": "Prague sand ST55",
        "467-01_Glassbeads_200_300": "GFZ, GB (200-300µm)",
        "468-01_Feldsparsand_Florence": "Florence feldspar sand",
        "491-01_Glassbeads_200_300_Prague": "Prague (200-300µm)",
        "494-01_Quartzsand_Uni_Freiburg": "Freiburg sand",
    }
    annot_mat = {
        "02_BernNeusand_1kPa": "Bern sand (2015)",
        "03_UtrechtSand2018_1kPa": "Utrecht sand",
        "04_BernNeusand_20kPa": "Bern sand (2015)",
        "05_RennesSandNE342021_1kPa": "Rennes, NE34",
        "06_ParmaSand_1kPa": "Parma sand",
        "07_KyotoSand_1kPa": "Kyoto sand",
        "08_BernSand2008_1kPa": "Bern sand (2008)",
        "09_GFZKorundNKF120_1kPa": "GFZ, NKF120",
        "10_UppsalaSand_1kPa": "Uppsala sand",
        "11_CergyPontoiseSand_1kPa": "Cergy Pontoise sand",
        "12_KorundsandBern2019_1kPa": "Bern, brown (2019)",
        "13_FeldspatsandUtrecht_1kPa": "Utrecht feldspar sand",
        "14_QuartzsandRHULondon_1kPa": "RHU London sand",
        "15b_G12GFZsmaller400_1kPa": "GFZ, G12",
        "15_G12GFZsmaller400_1kPa": "GFZ, G12",
        "16_G23GFZsmaller630_1kPa": "GFZ, G23",
        "17_QuartzsandPragST55_1kPa": "Prag, ST55",
        "18_GarnetSandGFZ2008_1kPa": "GFZ garnet sand",
        "19_Glasperlen70-110GFZ_1kPa": "GFZ, GB (70-110µm)",
        "20_Glasperlen300-400_1kPa": "GFZ, GB (300-400µm)",
        "21_Glasperlen40-70GFZ_1kPa": "GFZ, GB (40-70µm)",
        "22_Glasperlen100-200GFZ_1kPa": "GFZ, GB (100-200µm)",
        "23_ZirkonsandGFZ2009_1kPa": "GFZ zircon sand",
        "24_IFPKorundWeissBern2008_1kPa": "Bern, white (2008)",
        "25_GlasbeadsPrag0-50poured_1kPa": "Prag, GB (0-50µm)",
        "26_GlasbeadsPrag100-200_1kPa": "Prag, GB (100-200µm)",
        "28_SandBernCarloAG100-300_1kPa": "Bern, CarloAG (100-300µm)",
        "29_BernNeusand2015_5kPa": "Bern sand (2015)",
        "30_BernNeusand2015_10kPa": "Bern sand (2015)",
        "31_BernNeusand2015_15kPa": "Bern sand (2015)",
        "35_FS900SBernFspSand_1kPa": "Bern, FS900S",
        "36_LilleQtzSandG39_1kPa": "Lille, GA39",
        "37_LilleQtzSandNE34_1kPa": "Lille, NE34",
        "38_LilleGBCVP70-110_1kPa": "Lille, CVP (70-110µm)",
        "39_LilleGBEyraud70-110_1kPa": "Lille, Eyraud (70-110µm)",
        "40_PragueQtzSand_1kPa": "Prag sand",
        "41_WroclawQtzSand_1kPa": "Wroclaw sand",
        "42_TaiwanQtzSand_1kPa": "Taiwan sand",
        "43_OuroPretoQtzSand_1kPa": "Ouro Preto sand",
        "45_BernKorundSand_1kPa": "Bern, brown (2008)",
        "46_FreiburgQtzSand_1kPa": "Freiburg sand",
        "47b_GFZFoamglass250-500_1kPa": "GFZ, foam (250-500µm)",
        "392-01_Foamglas_0,25-0,5": "GFZ, foam (250-500µm)",
        "393-01_Foamglas_0,1-0,3": "GFZ, foam (100-300µm)",
        "47_GFZFoamglass250-500_1kPa": "GFZ, foam (250-500µm)",
        "48_BernNeusand_1kPa": "Bern sand (2015)",
        "49_GFZFoamglass100-300_1kPa": "GFZ, foam (100-300µm)",
        "50_BernNeusand_15kPa": "Bern sand (2015)",
        "071-01_Korund_B_MK": "Bern corundum brown",
        "101-01_Granatsand_GFZ": "GFZ garnet sand",
        "102-01_PB_Bernsand": "Bern sand (2008)",
        "103-01_PB_Lille1": "Lille, NE34",
        "105-01_PB_Taiwan1": "Taiwan sand",
        "108-01_PB_Cergy_Pont": "Cergy Pontoise sand",
        "109-01_PB_Royal_S": "RHU London sand",
        "110-01_PB_Uppsala": "Uppsala sand",
        "111-01_PB_Kyoto_S": "Kyoto sand",
        "112-01_PB_Ouro_Preto": "Ouro Preto sand",
        "113-01_PB_Parma_S": "Parma sand",
        "114-01_PB_Bernsand2": "Bern sand (2008)",
        "115-01_PB_Bernsand3": "Bern sand (2008)",
        "129-01_PB_GFZ_Sand": "GFZ sand",
        "137-01_PB_IFP_C_wi": "IFP corundum sand white",
        "140-01_PB_IFP_C_br": "IFP corundum sand brown",
        "150-01 Bernsand": "Bern sand (2008)",
        "153-01_PB_Pragsand": "Prag sand",
        "155-01_PB_Wroclaws": "Wroclaw sand",
        "159-01_PB_Zirkon": "GFZ zircon sand",
        "163-01_PB_GfzGran": "GFZ garnet sand",
        "165-01_PB_Glb_03-04": "GFZ, GB (300-400µm)",
        "167-01_PB_Glb_01-02": "GFZ, GB (100-200µm)",
        "169-01_PB_Glb_01-04": "GFZ, GB (100-400µm)",
        "176-01_G23(2009)": "GFZ, G23",
        "177-01_G23(2009)": "GFZ, G23",
        "178-01_G23(2009)": "GFZ, G23",
        "185-01 G23 2005": "GFZ, G23",
        "186-01-G23(2005)": "GFZ, G23",
        "187-01 G23(2005)": "GFZ, G23",
        "190-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "191-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "192-01 Glassbeads 200-300": "GFZ, GB (200-300µm)",
        "193-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "194-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "195-01 Glassbeads 40-70": "GFZ, GB (40-70µm)",
        "196-01 Glassbeads 100-200": "GFZ, GB (100-200µm)",
        "197-01 Glassbeads 70-110": "GFZ, GB (100-200µm)",
        "198-01 Glassbeads 100-200": "GFZ, GB (100-200µm)",
        "222-01_G23Sand2010": "GFZ, G23",
        "223-01_G23Sand2009": "GFZ, G23",
        "224-01_Glassbeads300-400": "GFZ, GB (300-400µm)",
        "226-01_GB100-200": "GFZ, GB (100-200µm)",
        "227-01_GB100-200": "GFZ, GB (100-200µm)",
        "228-01_GB300-400": "GFZ, GB (300-400µm)",
        "230-01_GB300-400": "GFZ, GB (300-400µm)",
        "231-01_GB70-110": "GFZ, GB (70-110µm)",
        "232-01_GB70-110": "GFZ, GB (70-110µm)",
        "233-01_GB40-70": "GFZ, GB (40-70µm)",
        "242-01 GB 40-70": "GFZ, GB (40-70µm)",
        "243-01 GB 200-300": "GFZ, GB (200-300µm)",
        "244-01 GB 200-300": "GFZ, GB (200-300µm)",
        "251-01 GB 40-70": "GFZ, GB (40-70µm)",
        "252-01 GB 70-110": "GFZ, GB (70-110µm)",
        "253-01 GB 100-200": "GFZ, GB (100-200µm)",
        "254-01 GB 200-300": "GFZ, GB (200-300µm)",
        "255-01 GB 300-400": "GFZ, GB (300-400µm)",
        "270-01 Granatsand": "GFZ garnet sand",
        "281-01_VUBSand_SamPoppe": "VUB sand",
        "286-01_VUB-Sand_Sieved_SamPoppe": "VUB sand",
        "323-01_G23": "GFZ, G23",
        "352-01_Glasperlen_100-200": "GFZ, GB (100-200µm)",
        "355-01_Glasperlen_40-70": "GFZ, GB (40-70µm)",
        "356-01_Glasperlen_200-300": "GFZ, GB (200-300µm)",
        "357_01_Glasperlen_300-400": "GFZ, GB (300-400µm)",
        "362-01_Glasperlen_70-110": "GFZ, GB (70-110µm)",
        "373-01_DB2017-X1_40-60_0.3-0.45mm": "Chengdu X1",
        "374-01_DB2017-X2_40-60_0.2-0.3mm": "Chengdu X2",
        "377-01 DB2017-L1_40-60_0.3-0.45mm": "Chengdu L1",
        "378-01_DB2017-L2_60-80_0.2-0.3mm": "Chengdu L2",
        "385-01_Zaragoza_Quartz_Sand": "Zaragoza sand",
        "390-01_Feldspar_sand_Utrecht": "Utrecht feldspar sand",
        "391-01_Quartz_sand_Utrecht": "Utrecht sand",
        "399-01_G12": "GFZ, G12",
        "400-01_G23": "GFZ, G23",
        "419-01_Quartzsand_Zaragosa_L-7080S": "Zaragoza sand 7080",
        "424-01_G12": "GFZ, G12",
        "426-01_G23": "GFZ, G23",
        "431-01_Qtz-Sand Fort_Rennes": "Rennes, NE34",
        "432-01_VUB_Sand_Belgium": "VUB sand",
        "435-01_Quartz_Sand_Bern": "Bern sand (2015)",
        "436-01_Korundsand_NFK120": "GFZ, NKF120",
        "437-01_Glassbeads_300_400": "GFZ, GB (300-400µm)",
        "439-01_Glassbeads_200_300": "GFZ, GB (200-300µm)",
        "440-01_Granatsand": "GFZ garnet sand",
        "441-01_Glassbeads_100_200": "GFZ, GB (100-200µm)",
        "442-01_Glassbeads_40_70": "GFZ, GB (40-70µm)",
        "443-03_Bernsand_30cm": "Bern sand (2015)",
        "444-03_Bernkorund_30cm": "Bern, brown (2019)",
        "446-01_Glassbeads_70_110": "GFZ, GB (70-110µm)",
        "452-01_Quartzsand_ST55_Prague": "Prag, ST55",
        "467-01_Glassbeads_200_300": "GFZ, GB (200-300µm)",
        "468-01_Feldsparsand_Florence": "Florence feldspar sand",
        "491-01_Glassbeads_200_300_Prague": "Prag, GB (200-300µm)",
        "494-01_Quartzsand_Uni_Freiburg": "Freiburg sand",
    }
    qualiscore = {
        "02_BernNeusand_1kPa": 6.9,
        "03_UtrechtSand2018_1kPa": 7.2,
        "05_RennesSandNE342021_1kPa": 5.5,
        "06_ParmaSand_1kPa": 6.1,
        "07_KyotoSand_1kPa": 9.1,
        "08_BernSand2008_1kPa": 7.5,
        "09_GFZKorundNKF120_1kPa": 9,
        "10_UppsalaSand_1kPa": 9.2,
        "11_CergyPontoiseSand_1kPa": 5.5,
        "12_KorundsandBern2019_1kPa": 9.7,
        "13_FeldspatsandUtrecht_1kPa": 11.15,
        "14_QuartzsandRHULondon_1kPa": 6.5,
        "15_G12GFZsmaller400_1kPa": 4.9,
        "16_G23GFZsmaller630_1kPa": 5.9,
        "17_QuartzsandPragST55_1kPa": 8.8,
        "18_GarnetSandGFZ2008_1kPa": 8.3,
        "19_Glasperlen70-110GFZ_1kPa": 3,
        "20_Glasperlen300-400_1kPa": 3,
        "21_Glasperlen40-70GFZ_1kPa": 3,
        "22_Glasperlen100-200GFZ_1kPa": 4.2,
        "23_ZirkonsandGFZ2009_1kPa": 6.6,
        "25_GlasbeadsPrag0-50poured_1kPa": 4,
        "26_GlasbeadsPrag100-200_1kPa": 3,
        "28_SandBernCarloAG100-300_1kPa": 7,
        "35_FS900SBernFspSand_1kPa": 11.1,
        "36_LilleQtzSandG39_1kPa": 8.8,
        "37_LilleQtzSandNE34_1kPa": 5.5,
        "38_LilleGBCVP70-110_1kPa": 4.6,
        "39_LilleGBEyraud70-110_1kPa": 4.6,
        "40_PragueQtzSand_1kPa": 8.4,
        "41_WroclawQtzSand_1kPa": 9.6,
        "42_TaiwanQtzSand_1kPa": 7.4,
        "43_OuroPretoQtzSand_1kPa": 9.6,
        "45_BernKorundSand_1kPa": 9.8,
        "46_FreiburgQtzSand_1kPa": 7.1,
        "47b_GFZFoamglass250-500_1kPa": 5.6,
        # "Florence Fsp Sand": 10.9,
        # "IGCAS GB 200-300": 3,
        # "Rennes Sand": 6,
    }
    mass = {
        "02_BernNeusand_1kPa": 3599.5,
        "03_UtrechtSand2018_1kPa": 3674.0,
        "05_RennesSandNE342021_1kPa": 3697.6,
        "06_ParmaSand_1kPa": 3713.0,
        "07_KyotoSand_1kPa": 3615.1,
        "08_BernSand2008_1kPa": 3604.7,
        "09_GFZKorundNKF120_1kPa": 4096.2,
        "10_UppsalaSand_1kPa": 3627.4,
        "11_CergyPontoiseSand_1kPa": 3717.3,
        "12_KorundsandBern2019_1kPa": 4017.0,
        "13_FeldspatsandUtrecht_1kPa": 3382.2,
        "14_QuartzsandRHULondon_1kPa": np.mean([3607.8, 3613.0]),
        "15_G12GFZsmaller400_1kPa": np.mean([3666.3, 3669.9]),
        "16_G23GFZsmaller630_1kPa": 3702.2,
        "17_QuartzsandPragST55_1kPa": np.mean([3532, 3544.7, 3640.9]),
        "18_GarnetSandGFZ2008_1kPa": 4272.5,
        "19_Glasperlen70-110GFZ_1kPa": 3465.1,
        "20_Glasperlen300-400_1kPa": 3603.6,
        "21_Glasperlen40-70GFZ_1kPa": 3540.0,
        "22_Glasperlen100-200GFZ_1kPa": 3619.8,
        "23_ZirkonsandGFZ2009_1kPa": 4755.1,
        "25_GlasbeadsPrag0-50poured_1kPa": 3540.5,
        "26_GlasbeadsPrag100-200_1kPa": 3594.0,
        "28_SandBernCarloAG100-300_1kPa": 3630.5,
        "29_BernNeusand2015_5kPa": 3628.4,
        "30_BernNeusand2015_10kPa": 3632.4,
        "31_BernNeusand2015_15kPa": 3540.2,
        "35_FS900SBernFspSand_1kPa": 3310.1,
        "36_LilleQtzSandG39_1kPa": 3480.4,
        "37_LilleQtzSandNE34_1kPa": 3641.5,
        "38_LilleGBCVP70-110_1kPa": 3536.9,
        "39_LilleGBEyraud70-110_1kPa": 3560.4,
        "40_PragueQtzSand_1kPa": 3556.9,
        "41_WroclawQtzSand_1kPa": 3401.5,
        "42_TaiwanQtzSand_1kPa": 3603.8,
        "43_OuroPretoQtzSand_1kPa": 3534.7,
        "45_BernKorundSand_1kPa": 3835.7,
        "46_FreiburgQtzSand_1kPa": 3642.1,
        "47b_GFZFoamglass250-500_1kPa": 2481.4,
        "48_BernNeusand_1kPa": 3554.2,
        "49_GFZFoamglass100-300_1kPa": 2645.2
        # "Florence Fsp Sand": 10.9,
        # "IGCAS GB 200-300": 3,
        # "Rennes Sand": 6,
    }
    group_dict = {
        "mats": mats,
        "mat_group": mat_group,
        "annot": annot,
        "qualiscore": qualiscore,
        "mass": mass,
        "annot_mat": annot_mat,
    }

    return group_dict


def get_outcasts() -> dict:
    """Samples that are of special attention"""
    outcasts = {
        "25_GlasbeadsPrag0-50poured_1kPa": "abnormally high healing values",
        "29_BernNeusand2015_5kPa": "non-standard normal stress",
        "30_BernNeusand2015_10kPa": "non-standard normal stress",
        "31_BernNeusand2015_15kPa": "non-standard normal stress",
        "26_GlasbeadsPrag100-200_1kPa": "abnormally high compaction",
    }
    return outcasts


def get_benchmarks() -> list:
    """Samples that were in the 2008 Klinkmüller benchmark"""
    benchmarks = [
        "40_PragueQtzSand_1kPa",
        "06_ParmaSand_1kPa",
        "07_KyotoSand_1kPa",
        "08_BernSand2008_1kPa",
        "10_UppsalaSand_1kPa",
        "11_CergyPontoiseSand_1kPa",
        "14_QuartzsandRHULondon_1kPa",
        "15b_G12GFZsmaller400_1kPa",
        "15_G12GFZsmaller400_1kPa",
        "18_GarnetSandGFZ2008_1kPa",
        "20_Glasperlen300-400_1kPa",
        "22_Glasperlen100-200GFZ_1kPa",
        "23_ZirkonsandGFZ2009_1kPa",
        "24_IFPKorundWeissBern2008_1kPa",
        "37_LilleQtzSandNE34_1kPa",
        "41_WroclawQtzSand_1kPa",
        "42_TaiwanQtzSand_1kPa",
        "43_OuroPretoQtzSand_1kPa",
        "101-01_Granatsand_GFZ",
        "102-01_PB_Bernsand",
        "103-01_PB_Lille1",
        "105-01_PB_Taiwan1",
        "108-01_PB_Cergy_Pont",
        "109-01_PB_Royal_S",
        "110-01_PB_Uppsala",
        "111-01_PB_Kyoto_S",
        "112-01_PB_Ouro_Preto",
        "113-01_PB_Parma_S",
        "114-01_PB_Bernsand2",
        "115-01_PB_Bernsand3",
        "129-01_PB_GFZ_Sand",
        "137-01_PB_IFP_C_wi",
        "140-01_PB_IFP_C_br",
        "150-01 Bernsand",
        "153-01_PB_Pragsand",
        "155-01_PB_Wroclaws",
        "159-01_PB_Zirkon",
        "163-01_PB_GfzGran",
        "165-01_PB_Glb_03-04",
        "167-01_PB_Glb_01-02",
        "169-01_PB_Glb_01-04",
    ]
    return benchmarks
