#!/usr/bin/env python3
# This file is part of granularhealing.

# granularhealing is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# granularhealing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with granularhealing. If not, see <https://www.gnu.org/licenses/>.

"""
Contains functions to batch analyze json files output by rst-evaluation
"""

import json
import os

import animation
import granularhealing.angles as ghangles
import granularhealing.files as ghfiles
import granularhealing.plots as ghplots
import numpy as np
from uncertainties.unumpy import uarray


def main():
    base_folder = ghfiles.ask_for_folder()

    file_list = [
        os.path.join(base_folder, f)
        for f in os.listdir(base_folder)
        if f.endswith(".json") and not ("all_data" in f)
    ]
    all_data = plot_all_data(file_list, base_folder)
    # all_data = plot_summarized_data(file_list, base_folder)
    save_all_data(base_folder, "000_all_data.json", all_data)


def save_all_data(base_path, fname, all_data):
    """Stores data into a json file for other plots"""
    save_path = os.path.join(base_path, fname)
    with open(save_path, "wt") as json_file:
        json.dump(all_data, json_file, sort_keys=True, indent=4)


@animation.wait()
def plot_summarized_data(file_list: list, base_folder: os.PathLike):
    """
    Cumulates data for the same material based on the description (e.g. all GFZ
    G12 sands)
    """

    all_data = get_all_data(file_list)
    unique_descript = np.unique(all_data["txt"])
    # print(unique_descript)
    return all_data


def get_all_data(file_list: list) -> dict:
    """Loads all data into a dictionary"""
    base_folder, _ = os.path.split(file_list[0])
    data_file = os.path.join(base_folder, "000_all_data.json")
    if os.path.exists(data_file):
        with open(data_file, "rt") as json_file:
            all_data = json.load(json_file)
    else:
        all_data = {
            "name": [],
            "mat": [],
            "txt": [],
            "mup": [],
            "mud": [],
            "mur": [],
            "cp": [],
            "cd": [],
            "cr": [],
            "mupe": [],
            "mude": [],
            "mure": [],
            "cpe": [],
            "cde": [],
            "cre": [],
            "picks": [],
        }

        for file_path in file_list:
            _, fname = os.path.split(file_path)
            name = fname.replace("_pick_data.json", "")
            all_data["name"].append(name)
            group_dict = ghgroups.get_group_dict()
            all_data["mat"].append(group_dict["mat_group"][name])
            all_data["txt"].append(group_dict["annot"][name])
            add_data(file_path, all_data)
    return all_data


@animation.wait()
def plot_all_data(file_list: list, base_folder: os.PathLike):
    """Plots all data without summarizing similar materials"""
    all_data = get_all_data(file_list)

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["mup"],
        all_data["mupe"],
        xlabel=r"peak strength $\mu_p$ ()",
        fname="000_AllRSTEval_peaks_friction",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["mud"],
        all_data["mude"],
        xlabel=r"dynamic strength $\mu_d$ ()",
        fname="000_AllRSTEval_dynamic_friction",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["mur"],
        all_data["mure"],
        xlabel=r"reactivation strength $\mu_r$ ()",
        fname="000_AllRSTEval_reactivation_friction",
    )

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        ghangles.fric_ang(all_data["mup"]),
        ghangles.fric_ang(all_data["mupe"]),
        xlabel=r"peak angle $\alpha_p$ (°)",
        fname="000_AllRSTEval_peaks_angles",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        ghangles.fric_ang(all_data["mud"]),
        ghangles.fric_ang(all_data["mude"]),
        xlabel=r"dynamic angle $\alpha_d$ (°)",
        fname="000_AllRSTEval_dynamic_angles",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        ghangles.fric_ang(all_data["mur"]),
        ghangles.fric_ang(all_data["mure"]),
        xlabel=r"reactivation angle $\alpha_r$ (°)",
        fname="000_AllRSTEval_reactivation_angles",
    )

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["cp"],
        all_data["cpe"],
        xlabel=r"peak cohesion $C_p$ (Pa)",
        fname="000_AllRSTEval_peaks_cohesion",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["cd"],
        all_data["cde"],
        xlabel=r"dynamic cohesion $C_d$ (Pa)",
        fname="000_AllRSTEval_dynamic_cohesion",
    )
    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        all_data["cr"],
        all_data["cre"],
        xlabel=r"reactivation cohesion $C_r$ (Pa)",
        fname="000_AllRSTEval_reactivation_cohesion",
    )

    mup = uarray(all_data["mup"], all_data["mupe"])
    mur = uarray(all_data["mur"], all_data["mure"])
    mudif = mup - mur

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        np.array([m.n for m in mudif]),
        np.array([m.s for m in mudif]),
        xlabel=r"friction difference $\mu_p - \mu_r$ ()",
        fname="000_AllRSTEval_difference_friction",
    )

    cp = uarray(all_data["cp"], all_data["cpe"])
    cr = uarray(all_data["cr"], all_data["cre"])
    cdif = cp - cr

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        np.array([m.n for m in cdif]),
        np.array([m.s for m in cdif]),
        xlabel=r"cohesion difference $C_p - C_r$ (Pa)",
        fname="000_AllRSTEval_difference_cohesion",
    )
    mudif = 100 * (mur / mup)

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        np.array([m.n for m in mudif]),
        np.array([m.s for m in mudif]),
        xlabel=r"friction change $\frac{\mu_r}{\mu_p}$ (%)",
        fname="000_AllRSTEval_ratio_friction",
    )

    cdif = 100 * (cp / cr)

    ghplots.create_overview_plot(
        base_folder,
        all_data["name"],
        np.array([m.n for m in cdif]),
        np.array([m.s for m in cdif]),
        xlabel=r"cohesion change $\frac{C_p}{C_r}$ (%)",
        fname="000_AllRSTEval_ratio_cohesion",
    )
    return all_data


def add_data(file_path: os.PathLike, all_data: dict) -> None:
    """Gets data from file for plotting"""
    with open(file_path, "rt") as json_file:
        data = json.load(json_file)
        all_data["mup"].append(data["fric_std"][0][0])
        all_data["mupe"].append(data["fric_std"][0][1])
        all_data["cp"].append(data["fric_std"][0][2])
        all_data["cpe"].append(data["fric_std"][0][3])
        all_data["mud"].append(data["fric_std"][1][0])
        all_data["mude"].append(data["fric_std"][1][1])
        all_data["cd"].append(data["fric_std"][1][2])
        all_data["cde"].append(data["fric_std"][1][3])
        all_data["mur"].append(data["fric_std"][2][0])
        all_data["mure"].append(data["fric_std"][2][1])
        all_data["cr"].append(data["fric_std"][2][2])
        all_data["cre"].append(data["fric_std"][2][3])
        all_data["picks"].append(data["picks"])


if __name__ == "__main__":
    main()
