#!/usr/bin/env python3
# This file is part of granularhealing.

# granularhealing is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# granularhealing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with granularhealing. If not, see <https://www.gnu.org/licenses/>.
import json
import os

import granularhealing.cfit as ghcfit
import granularhealing.files as ghfiles
import granularhealing.plots as ghplot
import matplotlib.pyplot as plt
import numpy as np
from matplotlib.offsetbox import AnchoredText


def main():
    meas_file_path = ghfiles.ask_for_file()
    base_path, fname = os.path.split(meas_file_path)
    fname, _ = os.path.splitext(fname)
    exp = ghfiles.open_tdms(meas_file_path)

    fric = exp["data"]["shear"] / exp["data"]["normal"]

    # peaks = [
    #     (150, 160),
    #     (400, 410),
    #     (995, 1005),
    #     (2595, 2605),
    #     (7695, 7705),
    #     (22800, 22810),
    #     (72895, 72905)
    # ]
    pick_data = get_picks(meas_file_path.replace("tdms", "json"), exp)
    stable = np.mean(fric[pick_data["stable_start"] : pick_data["stable_end"]])
    picks = [
        ii[0] + np.argmax(fric[ii[0] : ii[1]]) for ii in pick_data["peaks"]
    ]

    mudiff = fric[picks] - stable
    times = exp["data"]["time"][picks]
    init = [exp["data"]["time"][pick_data["stable_end"]]]
    init.extend(times)
    dttimes = np.diff(init)

    fig, ax = create_plot(dttimes, mudiff)
    fig.savefig(os.path.join(base_path, "ManualSHS_%s.png" % fname))
    fig.savefig(os.path.join(base_path, "ManualSHS_%s.pdf" % fname))
    # plt.show()


def get_picks(json_path, exp):
    """
    Tries to read manual picks from an existing json file. If not creates a
    plot to manually pick values.
    """

    if os.path.exists(json_path):
        with open(json_path, "rt") as json_file:
            pick_data = json.load(json_file)
    else:
        plot_ts(exp)
        pick_data = {"peaks": []}
        pick_data["stable_start"] = int(
            input("Enter start of stable sliding: ")
        )
        pick_data["stable_end"] = int(input("Enter end of stable sliding: "))
        picking = True
        while picking:
            start_area = input("Enter start area: ")
            end_area = input("Enter end area: ")
            if start_area:
                pick_data["peaks"].append((int(start_area), int(end_area)))
            else:
                picking = False
        with open(json_path, "wt") as json_file:
            json.dump(pick_data, json_file, sort_keys=True, indent=4)
    return pick_data


def create_plot(ht, mud):
    fig, ax = plt.subplots()
    ax.semilogx(ht, mud, "s", base=10, label="data")

    (xq, yq, ucb, lcb, popt, perr) = ghplot.get_fit(np.log10(ht), mud)

    ax.semilogx(10**xq, yq, color="C1", label="fit")
    ax.semilogx(10**xq, lcb, color="C1", linestyle=":", label="95% confband")
    ax.semilogx(10**xq, ucb, color="C1", linestyle=":")

    ax.legend()
    ax.add_artist(
        AnchoredText(
            "b = %s" % ghcfit.sign_str(popt[0], perr[0]), loc="lower right"
        )
    )

    ax.set_xlabel(r"hold time $t_h$ (s)")
    ax.set_ylabel(r"apparent friction change $\Delta\mu_p$")
    fig.tight_layout()
    return fig, ax


def plot_ts(exp):
    load_disp = np.cumsum(exp["data"]["velocity"] * exp["dt"])
    fig, axes = plt.subplots(nrows=2, sharex=True)
    axes[0].plot(exp["data"]["shear"])
    axes[1].plot(exp["data"]["velocity"])

    plt.show(block=False)


if __name__ == "__main__":
    main()
